#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_AntigenTyping : BaseTest 
	{
		
		private DataTable _dtTests;
		private DataRow _drRHNeg;
		private DataRow _drRHPos;
		private System.Guid _patientGuid;
		private System.Guid _bpGuid;

		[SetUp]
		protected void SetUp()		
		{
			this.BaseSetUp();

			if (this.RefreshData)
			{

				this._dtTests = new DataTable();
				this._dtTests.Columns.Add(TABLES.BloodTestType.OrderableTestId,typeof(int));
				this._dtTests.Columns.Add(TABLES.BloodUnitMedia.BloodUnitGuid,typeof(Guid));
				this._dtTests.Columns.Add(ARTIFICIAL.Interp);

				DataTable dtData = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM BloodUnitMedia WHERE RhFactorCode = 'P'");
				if (dtData.Rows.Count == 0)
				{	
					DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, Common.ComponentClass.RBC).BloodProductGuid,Common.ABO.A, Common.RH.Positive, this.TestDivisionCode, true);
					dtData = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM BloodUnitMedia WHERE RhFactorCode = 'P'", true);
				}

				this._drRHPos = this._dtTests.NewRow();
				this._drRHPos[TABLES.BloodUnitMedia.BloodUnitGuid] = (System.Guid)dtData.Rows[0][TABLES.BloodUnitMedia.BloodUnitGuid];

				dtData = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM BloodUnitMedia WHERE RhFactorCode = 'N'");
				if (dtData.Rows.Count == 0)
				{	
					DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, DataCreator.GetRandomBloodProduct(true, Common.ComponentClass.RBC).BloodProductGuid,Common.ABO.A, Common.RH.Negative, this.TestDivisionCode, true);
					dtData = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodUnitGuid FROM BloodUnitMedia WHERE RhFactorCode = 'N'", true);
				}

				this._drRHNeg = this._dtTests.NewRow();
				this._drRHNeg[TABLES.BloodUnitMedia.BloodUnitGuid] = (System.Guid)dtData.Rows[0][TABLES.BloodUnitMedia.BloodUnitGuid];

				this._patientGuid = DataCreator.GetRandomRecordGuid(TABLES.Patient.TableName, "");
				this._bpGuid = DataCreator.GetRandomRecordGuid(TABLES.BloodProduct.TableName, "ProductTypeCode = 'E002' AND ProductCodeIndicator = 0");

				this.RefreshData = false;
			}
		}


		[Test]
		public void InitializeLookup_Pass()
		{
			System.Collections.Hashtable ht = BOL.AntigenTyping.InitializeLookup();
			//
			Assert.AreEqual(ht[""].ToString(), "X", "X1");
			Assert.AreEqual(ht["X"].ToString(), "X", "X2");

			Assert.AreEqual(ht["1"].ToString() ,"P", "P1");
			Assert.AreEqual(ht["2"].ToString(), "P", "P2");
			Assert.AreEqual(ht["3"].ToString(), "P", "P3");
			Assert.AreEqual(ht["4"].ToString(), "P", "P4");
			Assert.AreEqual(ht["P"].ToString(), "P", "P5");
			//Used to be inconclusive - CR2653
			Assert.AreEqual(ht["W"].ToString(), "P", "P6");
			Assert.AreEqual(ht["M"].ToString(), "P", "P7");
			Assert.AreEqual(ht["F"].ToString(), "P", "P8");

			Assert.AreEqual(ht["R"].ToString(), "N", "N1");
			Assert.AreEqual(ht["0"].ToString(), "N", "N2");
			Assert.AreEqual(ht["N"].ToString(), "N", "N3");

		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void InitializeLookup_Fail()
		{
		}

		[Test]
		public void InterpretPhenotypeResults_DataRow_Pass()
		{
			string interp;
			//
			DataTable dt = new DataTable();
			dt.Columns.Add(ARTIFICIAL.IS);
			dt.Columns.Add(ARTIFICIAL.RT);
			dt.Columns.Add(ARTIFICIAL.C37);
			dt.Columns.Add(ARTIFICIAL.AHG);
			dt.Columns.Add(ARTIFICIAL.CC);
			dt.Columns.Add(TABLES.ReagentTestParameter.ReactivityPhaseCode,typeof(int));
			//
			DataRow drISOnly = dt.NewRow();
			drISOnly[TABLES.ReagentTestParameter.ReactivityPhaseCode] = (int)Common.ReactivityPhase.IS;
			DataRow drISRT = dt.NewRow();
			drISRT[TABLES.ReagentTestParameter.ReactivityPhaseCode] = (int)Common.ReactivityPhase.ISandRT;
			DataRow drIS37 = dt.NewRow();
			drIS37[TABLES.ReagentTestParameter.ReactivityPhaseCode] = (int)Common.ReactivityPhase.ISand37;
			DataRow drAHG = dt.NewRow();
			drAHG[TABLES.ReagentTestParameter.ReactivityPhaseCode] = (int)Common.ReactivityPhase.AHGandCC;
			//
			drISOnly[ARTIFICIAL.IS] = "0";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISOnly);
			Assert.AreEqual("N", interp,"N1");
			//
			drISOnly[ARTIFICIAL.IS] = "2";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISOnly);
			Assert.AreEqual("P", interp,"P1");
			//
			drISOnly[ARTIFICIAL.IS] = "M";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISOnly);
			Assert.AreEqual("P", interp,"P2");
			//
			drISOnly[ARTIFICIAL.IS] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISOnly);
			Assert.AreEqual("X", interp,"X1");
			//
			drISOnly[ARTIFICIAL.IS] = "";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISOnly);
			Assert.AreEqual("", interp,"X2");
			//
			//
			drISRT[ARTIFICIAL.IS] = "0";
			drISRT[ARTIFICIAL.RT] = "0";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("N", interp,"N2");
			//
			drISRT[ARTIFICIAL.IS] = "2";
			drISRT[ARTIFICIAL.RT] = "1";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("P", interp,"P3");
			//
			drISRT[ARTIFICIAL.IS] = "0";
			drISRT[ARTIFICIAL.RT] = "2";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("P", interp,"P4");
			//
			drISRT[ARTIFICIAL.IS] = "P";
			drISRT[ARTIFICIAL.RT] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("P", interp,"P5");
			//
			drISRT[ARTIFICIAL.IS] = "X";
			drISRT[ARTIFICIAL.RT] = "M";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("P", interp,"P6");
			//
			drISRT[ARTIFICIAL.IS] = "X";
			drISRT[ARTIFICIAL.RT] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drISRT);
			Assert.AreEqual("X", interp,"X3");
			//
			//
			drIS37[ARTIFICIAL.IS] = "0";
			drIS37[ARTIFICIAL.C37] = "N";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("N", interp,"N3");
			//
			drIS37[ARTIFICIAL.IS] = "X";
			drIS37[ARTIFICIAL.C37] = "R";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("N", interp,"N4");
			//
			drIS37[ARTIFICIAL.IS] = "N";
			drIS37[ARTIFICIAL.C37] = "1";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("P", interp,"P7");
			//
			drIS37[ARTIFICIAL.IS] = "0";
			drIS37[ARTIFICIAL.C37] = "2";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("P", interp,"P8");
			//
			drIS37[ARTIFICIAL.IS] = "2";
			drIS37[ARTIFICIAL.C37] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("P", interp,"P9");
			//
			drIS37[ARTIFICIAL.IS] = "X";
			drIS37[ARTIFICIAL.C37] = "P";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("P", interp,"P10");
			//
			drIS37[ARTIFICIAL.IS] = "0";
			drIS37[ARTIFICIAL.C37] = "M";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("P", interp,"P11");
			//
			drIS37[ARTIFICIAL.IS] = "X";
			drIS37[ARTIFICIAL.C37] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drIS37);
			Assert.AreEqual("X", interp,"X4");
			//
			//
			drAHG[ARTIFICIAL.AHG] = "0";
			drAHG[ARTIFICIAL.CC] = "1";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drAHG);
			Assert.AreEqual("N", interp,"N4");
			//
			drAHG[ARTIFICIAL.AHG] = "1";
			drAHG[ARTIFICIAL.CC] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drAHG);
			Assert.AreEqual("P", interp,"P12");
			//
			drAHG[ARTIFICIAL.AHG] = "M";
			drAHG[ARTIFICIAL.CC] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drAHG);
			Assert.AreEqual("P", interp,"P13");
			//
			drAHG[ARTIFICIAL.AHG] = "X";
			drAHG[ARTIFICIAL.CC] = "X";
			interp = BOL.AntigenTyping.InterpretPhenotypeResults(drAHG);
			Assert.AreEqual("X", interp,"X5");
		}

		[Test]
		public void InterpretPhenotypeResults_DataRow_Fail()
		{
			try
			{
				DataTable dt = new DataTable();
				DataRow dr = dt.NewRow();
				BOL.AntigenTyping.InterpretPhenotypeResults(dr);
				
				Assert.AreEqual(false, AssertionFailReasons.ArgumentExceptionExpected, "Empty DataRow");
			}
			catch(ArgumentException)
			{
				Assert.IsTrue(true, "Empty DataRow");
			}

			//Fail test #2--- Missing ReactivityPhaseCode
			DataTable dt2 = new DataTable();
			dt2.Columns.Add(ARTIFICIAL.IS);
			dt2.Columns.Add(ARTIFICIAL.RT);
			dt2.Columns.Add(ARTIFICIAL.C37);
			dt2.Columns.Add(ARTIFICIAL.AHG);
			dt2.Columns.Add(ARTIFICIAL.CC);
			dt2.Columns.Add(TABLES.ReagentTestParameter.ReactivityPhaseCode,typeof(int));
			DataRow drTest = dt2.NewRow();

			drTest[ARTIFICIAL.IS] = "X";
			drTest[ARTIFICIAL.RT] = "X";
			drTest[ARTIFICIAL.C37] = "X";
			drTest[ARTIFICIAL.AHG] = "X";
			drTest[ARTIFICIAL.CC] = "X";
			drTest[TABLES.ReagentTestParameter.ReactivityPhaseCode] = DBNull.Value;

			Assert.AreEqual(string.Empty, BOL.AntigenTyping.InterpretPhenotypeResults(drTest), "Null ReactivityPhaseCode");

		}

		[Test]
		public void InterpretWeakDResults_Pass()
		{
			string interp;
			string antiDInterp = "";
			string ControlDInterp = "";
			//
			interp = BOL.AntigenTyping.InterpretWeakDResults("2","2","X","X","X","X","I",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("I", interp,"I1");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("2","0","3","0","3","1","P",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("P", interp,"P1");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("X","X","0","0","1","1","N",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("N", interp,"N1");
			////////////////////////////////////////////////////////////////////////////////////////
			interp = BOL.AntigenTyping.InterpretWeakDResults("X","X","X","X","X","X","I",false, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("X", interp,"X1");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("2","","2","","2","","P",false, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("P", interp,"P2");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("0","0","0","0","1","0","N",false, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("N", interp,"N2");
			////////////////////////////////////////////////////////////////////////////////////////
			interp = BOL.AntigenTyping.InterpretWeakDResults("X","X","P","P","X","X","P",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("I", interp,"I2");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("0","0","P","P","X","X","P",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("I", interp,"I3");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("0","3","2","3","0","3","P",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("X", interp,"X2");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("2","","2","","2","","P",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("XX", interp,"X3");
			
			interp = BOL.AntigenTyping.InterpretWeakDResults("","","","","","","",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("", interp,"Blank");
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void InterpretWeakDResults_Fail()
		{
			string interp;
			string antiDInterp = "";
			string ControlDInterp = "";
			//
			interp = BOL.AntigenTyping.InterpretWeakDResults("Q","Q","Q","Q","Q","Q","Q",true, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("X", interp,"Garbage Data");
		}

		[Test]
		public void InterpretWeakDResults_DataRow_Pass()
		{
			string interp;
			string antiDInterp = "";
			string ControlDInterp = "";
			//
			DataTable dt = new DataTable();
			dt.Columns.Add(ARTIFICIAL.D37);
			dt.Columns.Add(ARTIFICIAL.Dc37);
			dt.Columns.Add(ARTIFICIAL.DAHG);
			dt.Columns.Add(ARTIFICIAL.DcAHG);
			dt.Columns.Add(ARTIFICIAL.DCC);
			dt.Columns.Add(ARTIFICIAL.DcCC);
			dt.Columns.Add(ARTIFICIAL.Interp);
			dt.Columns.Add(ARTIFICIAL.DControlReagentGuid,typeof(Guid));
			//
			DataRow drWeakD = dt.NewRow();
			//
			drWeakD[ARTIFICIAL.D37] = "2";
			drWeakD[ARTIFICIAL.Dc37] = "2";
			drWeakD[ARTIFICIAL.DAHG] = "X";
			drWeakD[ARTIFICIAL.DcAHG] = "X";
			drWeakD[ARTIFICIAL.DCC] = "X";
			drWeakD[ARTIFICIAL.DcCC] = "X";
			drWeakD[ARTIFICIAL.Interp] = "I";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.NewGuid();
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("I", interp,"I1");
			//
			drWeakD[ARTIFICIAL.D37] = "2";
			drWeakD[ARTIFICIAL.Dc37] = "0";
			drWeakD[ARTIFICIAL.DAHG] = "3";
			drWeakD[ARTIFICIAL.DcAHG] = "0";
			drWeakD[ARTIFICIAL.DCC] = "3";
			drWeakD[ARTIFICIAL.DcCC] = "1";
			drWeakD[ARTIFICIAL.Interp] = "P";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.NewGuid();
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("P", interp,"P1");
			//
			drWeakD[ARTIFICIAL.D37] = "X";
			drWeakD[ARTIFICIAL.Dc37] = "X";
			drWeakD[ARTIFICIAL.DAHG] = "0";
			drWeakD[ARTIFICIAL.DcAHG] = "0";
			drWeakD[ARTIFICIAL.DCC] = "1";
			drWeakD[ARTIFICIAL.DcCC] = "1";
			drWeakD[ARTIFICIAL.Interp] = "N";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.NewGuid();
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("N", interp,"N1");
			//
			drWeakD[ARTIFICIAL.D37] = "X";
			drWeakD[ARTIFICIAL.Dc37] = "X";
			drWeakD[ARTIFICIAL.DAHG] = "X";
			drWeakD[ARTIFICIAL.DcAHG] = "X";
			drWeakD[ARTIFICIAL.DCC] = "X";
			drWeakD[ARTIFICIAL.DcCC] = "X";
			drWeakD[ARTIFICIAL.Interp] = "I";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.Empty;
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("X", interp,"X1");
			//
			drWeakD[ARTIFICIAL.D37] = "2";
			drWeakD[ARTIFICIAL.Dc37] = "";
			drWeakD[ARTIFICIAL.DAHG] = "2";
			drWeakD[ARTIFICIAL.DcAHG] = "";
			drWeakD[ARTIFICIAL.DCC] = "2";
			drWeakD[ARTIFICIAL.DcCC] = "";
			drWeakD[ARTIFICIAL.Interp] = "P";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.Empty;
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("P", interp,"P2");
			//
			drWeakD[ARTIFICIAL.D37] = "0";
			drWeakD[ARTIFICIAL.Dc37] = "0";
			drWeakD[ARTIFICIAL.DAHG] = "0";
			drWeakD[ARTIFICIAL.DcAHG] = "0";
			drWeakD[ARTIFICIAL.DCC] = "1";
			drWeakD[ARTIFICIAL.DcCC] = "0";
			drWeakD[ARTIFICIAL.Interp] = "N";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.Empty;
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
			Assert.AreEqual("N", interp,"N2");
		}

		[Test]
		[ExpectedException(typeof(NullReferenceException))]
		public void InterpretWeakDResults_DataRow_Fail()
		{
			string interp;
			string antiDInterp = "";
			string ControlDInterp = "";
			//
			DataTable dt = new DataTable();
			dt.Columns.Add(ARTIFICIAL.D37);
			dt.Columns.Add(ARTIFICIAL.Dc37);
			dt.Columns.Add(ARTIFICIAL.DAHG);
			dt.Columns.Add(ARTIFICIAL.DcAHG);
			dt.Columns.Add(ARTIFICIAL.DCC);
			dt.Columns.Add(ARTIFICIAL.DcCC);
			dt.Columns.Add(ARTIFICIAL.Interp);
			dt.Columns.Add(ARTIFICIAL.DControlReagentGuid,typeof(Guid));
			//
			DataRow drWeakD = dt.NewRow();
			//
			drWeakD[ARTIFICIAL.D37] = "Q";
			drWeakD[ARTIFICIAL.Dc37] = "Q";
			drWeakD[ARTIFICIAL.DAHG] = "Q";
			drWeakD[ARTIFICIAL.DcAHG] = "Q";
			drWeakD[ARTIFICIAL.DCC] = "Q";
			drWeakD[ARTIFICIAL.DcCC] = "Q";
			drWeakD[ARTIFICIAL.Interp] = "Q";
			drWeakD[ARTIFICIAL.DControlReagentGuid] = Guid.NewGuid();
			interp = BOL.AntigenTyping.InterpretWeakDResults(drWeakD, ref antiDInterp, ref ControlDInterp);
		}

		[Test]
		public void InterpretWeakDResults_DataTable_Pass()
		{
			string message;
			//
			DataTable dt = new DataTable();
			dt.Columns.Add(ARTIFICIAL.C37);
			dt.Columns.Add(ARTIFICIAL.AHG);
			dt.Columns.Add(ARTIFICIAL.CC);
			dt.Columns.Add(ARTIFICIAL.Interp);
			dt.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.Indicator);
			//
			DataRow drAntiD = dt.NewRow();
			//
			drAntiD[ARTIFICIAL.C37] = "X";
			drAntiD[ARTIFICIAL.AHG] = "X";
			drAntiD[ARTIFICIAL.CC] = "X";
			drAntiD[ARTIFICIAL.Interp] = "X";
			dt.Rows.Add(drAntiD);
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("E", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"E1");
			Assert.AreEqual(Common.StrRes.Tooltips.Common.InvalidTestResults().ResString, message,"E1-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "2";
			drAntiD[ARTIFICIAL.AHG] = "2";
			drAntiD[ARTIFICIAL.CC] = "2";
			drAntiD[ARTIFICIAL.Interp] = "P";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("V", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"V1");
			Assert.AreEqual("", message,"V1-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "0";
			drAntiD[ARTIFICIAL.AHG] = "0";
			drAntiD[ARTIFICIAL.CC] = "1";
			drAntiD[ARTIFICIAL.Interp] = "N";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("V", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"V2");
			Assert.AreEqual("", message,"V2-Msg");
			//
			DataRow drDControl = dt.NewRow();
			//
			drAntiD[ARTIFICIAL.C37] = "2";
			drAntiD[ARTIFICIAL.AHG] = "X";
			drAntiD[ARTIFICIAL.CC] = "X";
			drAntiD[ARTIFICIAL.Interp] = "P";
			drDControl[ARTIFICIAL.C37] = "2";
			drDControl[ARTIFICIAL.AHG] = "X";
			drDControl[ARTIFICIAL.CC] = "X";
			drDControl[ARTIFICIAL.Interp] = "P";
			dt.Rows.Add(drDControl);
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("E", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"E2");
			Assert.AreEqual(Common.StrRes.Tooltips.Common.InterpretationNotMatch().ResString, message,"E2-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "2";
			drAntiD[ARTIFICIAL.AHG] = "3";
			drAntiD[ARTIFICIAL.CC] = "3";
			drAntiD[ARTIFICIAL.Interp] = "P";
			drDControl[ARTIFICIAL.C37] = "0";
			drDControl[ARTIFICIAL.AHG] = "0";
			drDControl[ARTIFICIAL.CC] = "1";
			drDControl[ARTIFICIAL.Interp] = "N";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("V", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"V3");
			Assert.AreEqual("", message,"V3-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "X";
			drAntiD[ARTIFICIAL.AHG] = "0";
			drAntiD[ARTIFICIAL.CC] = "1";
			drAntiD[ARTIFICIAL.Interp] = "N";
			drDControl[ARTIFICIAL.C37] = "X";
			drDControl[ARTIFICIAL.AHG] = "0";
			drDControl[ARTIFICIAL.CC] = "1";
			drDControl[ARTIFICIAL.Interp] = "N";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("V", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"V4");
			Assert.AreEqual("", message,"V4-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "2";
			drAntiD[ARTIFICIAL.AHG] = "2";
			drAntiD[ARTIFICIAL.CC] = "2";
			drAntiD[ARTIFICIAL.Interp] = "P";
			drDControl[ARTIFICIAL.C37] = "";
			drDControl[ARTIFICIAL.AHG] = "";
			drDControl[ARTIFICIAL.CC] = "";
			drDControl[ARTIFICIAL.Interp] = "";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("E", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"E3");
			Assert.AreEqual(Common.StrRes.Tooltips.Common.InvalidDControl().ResString, message,"E3-Msg");
			//
			drAntiD[ARTIFICIAL.C37] = "";
			drAntiD[ARTIFICIAL.AHG] = "";
			drAntiD[ARTIFICIAL.CC] = "";
			drAntiD[ARTIFICIAL.Interp] = "";
			drDControl[ARTIFICIAL.C37] = "";
			drDControl[ARTIFICIAL.AHG] = "";
			drDControl[ARTIFICIAL.CC] = "";
			drDControl[ARTIFICIAL.Interp] = "";
			message = BOL.AntigenTyping.InterpretWeakDResults(dt);
			Assert.AreEqual("", drAntiD[Common.DatabaseConstants.ArtificialColumnNames.Indicator],"Blank");
			Assert.AreEqual("", message,"Blank-Msg");
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void InterpretWeakDResults_DataTable_Fail()
		{
			DataTable dt = new DataTable();
			string interp = BOL.AntigenTyping.InterpretWeakDResults(dt);
		}

		[Test]
		public void CanUnitBeAntigenTyped_Pass()
		{
			BOL.BloodUnit unit = new BOL.BloodUnit(Guid.NewGuid());
			//
			//RED BLOOD CELLS
			unit.ProductType.ProductTypeCode = "E001";
			bool antigenType = BOL.AntigenTyping.CanUnitBeAntigenTyped(unit);
			Assert.AreEqual(true, antigenType, "RBC");
			//
			//PLASMA
			unit.ProductType.ProductTypeCode = "E017";
			antigenType = BOL.AntigenTyping.CanUnitBeAntigenTyped(unit);
			Assert.AreEqual(false, antigenType, "PLASMA");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CanUnitBeAntigenTyped_Fail()
		{
		}

		[Test]
		public void GetAntigenTypingTests_Pass()
		{
			DataTable dtTest = BOL.AntigenTyping.GetAntigenTypingTests();

			Assert.IsNotNull(dtTest);
		}

		[Test]
		[Ignore( StandardIgnoreReasons.NoFailureConditions )]
		public void GetAntigenTypingTests_Fail()
		{
		}

		[Test]
		public void WeakDPositiveOnRhNegativeUnit_Pass()
		{
			bool checkRh;
			//
			if (this._drRHNeg!=null)
			{
				this._drRHNeg[TABLES.BloodTestType.OrderableTestId] = (int)Common.OrderableTest.AGwD;
				this._drRHNeg[ARTIFICIAL.Interp] = "P";
				checkRh = BOL.AntigenTyping.WeakDPositiveOnRhNegativeUnit(this._drRHNeg,this._drRHNeg[ARTIFICIAL.Interp].ToString());
				Assert.AreEqual(true, checkRh, "Valid NegUnit Interp");
				//
				this._drRHNeg[ARTIFICIAL.Interp] = "N";
				checkRh = BOL.AntigenTyping.WeakDPositiveOnRhNegativeUnit(this._drRHNeg,this._drRHNeg[ARTIFICIAL.Interp].ToString());
				Assert.AreEqual(false, checkRh, "Invalid NegUnit Interp");
			}
			//
			if (this._drRHPos!=null)
			{
				this._drRHPos[TABLES.BloodTestType.OrderableTestId] = (int)Common.OrderableTest.AGwD;
				this._drRHPos[ARTIFICIAL.Interp] = "P";
				checkRh = BOL.AntigenTyping.WeakDPositiveOnRhNegativeUnit(this._drRHPos,this._drRHPos[ARTIFICIAL.Interp].ToString());
				Assert.AreEqual(false, checkRh, "Valid PosUnit Interp");
				//
				this._drRHPos[ARTIFICIAL.Interp] = "N";
				checkRh = BOL.AntigenTyping.WeakDPositiveOnRhNegativeUnit(this._drRHPos,this._drRHPos[ARTIFICIAL.Interp].ToString());
				Assert.AreEqual(false, checkRh, "Invalid PosUnit Interp");
			}
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void WeakDPositiveOnRhNegativeUnit_Fail()
		{
			DataTable dt = new DataTable();
			DataRow dr = dt.NewRow();
			BOL.AntigenTyping.WeakDPositiveOnRhNegativeUnit(dr,"");
		}

		[Test]
		public void AntigenTypingMismatch_BloodUnitTests_Pass()
		{
			System.Guid bloodUnitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), this._patientGuid, this._bpGuid, true);
			//
			//Insert a negative AGK interp.
			DataCreator.CreateBloodUnitTest(System.Guid.Empty, bloodUnitGuid, 260, "N");

			DataTable dtTest = new DataTable();
			dtTest.Columns.Add(TABLES.BloodTestType.OrderableTestId,typeof(int));
			dtTest.Columns.Add(TABLES.BloodUnitTest.BloodUnitGuid,typeof(System.Guid));
			dtTest.Columns.Add(TABLES.BloodUnitTest.DivisionCode, typeof(string));
			dtTest.Columns.Add(ARTIFICIAL.InterpGuid,typeof(System.Guid));
			dtTest.Columns.Add(ARTIFICIAL.Interp, typeof(string));

			DataRow drData = dtTest.NewRow();
			drData[TABLES.BloodUnitTest.BloodUnitGuid] = bloodUnitGuid;
			drData[TABLES.BloodTestType.OrderableTestId] = Convert.ToInt32(Common.OrderableTest.AGK);
			drData[TABLES.BloodUnitTest.DivisionCode] = this.TestDivisionCode;
			drData[ARTIFICIAL.InterpGuid] = Guid.NewGuid();
			drData[ARTIFICIAL.Interp] = "P";

			bool retValue = BOL.AntigenTyping.AntigenTypingMismatch(drData,drData[ARTIFICIAL.Interp].ToString());

			Assert.AreEqual(true, retValue);
		}

		[Test]
		public void AntigenTypingMismatch_BloodUnitTests_Fail_2()
		{
			System.Guid bloodUnitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), this._patientGuid, this._bpGuid, true);
			
			DataCreator.CreateBloodUnitTest(System.Guid.Empty, bloodUnitGuid, 320, "N");

			DataTable dtTest = new DataTable();
			dtTest.Columns.Add(TABLES.BloodTestType.OrderableTestId,typeof(int));
			dtTest.Columns.Add(TABLES.BloodUnitTest.BloodUnitGuid,typeof(System.Guid));
			dtTest.Columns.Add(TABLES.BloodUnitTest.DivisionCode, typeof(string));
			dtTest.Columns.Add(ARTIFICIAL.InterpGuid,typeof(System.Guid));
			dtTest.Columns.Add(ARTIFICIAL.Interp, typeof(string));

			DataRow drData = dtTest.NewRow();
			drData[TABLES.BloodUnitTest.BloodUnitGuid] = bloodUnitGuid;
			drData[TABLES.BloodTestType.OrderableTestId] = Convert.ToInt32(Common.OrderableTest.AGM);
			drData[TABLES.BloodUnitTest.DivisionCode] = this.TestDivisionCode;
			drData[ARTIFICIAL.InterpGuid] = Guid.NewGuid();
			drData[ARTIFICIAL.Interp] = "N";

			bool retValue = BOL.AntigenTyping.AntigenTypingMismatch(drData,drData[ARTIFICIAL.Interp].ToString());

			Assert.AreEqual(false, retValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void AntigenTypingMismatch_BloodUnitTests_Fail()
		{
			DataTable dt = new DataTable();
			DataRow dr = dt.NewRow();
			Assert.AreEqual(true, BOL.AntigenTyping.AntigenTypingMismatch(dr,""));
		}

		[Test]
		public void AntigenTypingMismatch_BloodUnitAntigens_Pass()
		{
			System.Guid bloodUnitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), this._patientGuid, this._bpGuid, true);
			//
			string insertBloodUnitAntigen = "INSERT INTO BloodUnitAntigen (";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.BloodUnitGuid + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.AntigenTypeId + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.Positive + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.DivisionCode + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.RecordStatusCode + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateUser + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateDate + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateFunctionId + ")";
			insertBloodUnitAntigen += " VALUES (";
			insertBloodUnitAntigen += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitAntigen += "'20',"; //E antigen
			insertBloodUnitAntigen += "0,"; //Negative
			insertBloodUnitAntigen += "'" + this.TestDivisionCode + "',";
			insertBloodUnitAntigen += "'" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString() + "',";
			insertBloodUnitAntigen += "'" + Common.LogonUser.LogonUserName + "',";
			insertBloodUnitAntigen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitAntigen += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			UnitTestUtil.RunSQL(insertBloodUnitAntigen);
			//
			DataTable dtTest = new DataTable();
			dtTest.Columns.Add(TABLES.BloodTestType.OrderableTestId,typeof(int));
			dtTest.Columns.Add(TABLES.BloodUnitTest.BloodUnitGuid,typeof(System.Guid));
			dtTest.Columns.Add(TABLES.BloodUnitTest.DivisionCode, typeof(string));
			dtTest.Columns.Add(ARTIFICIAL.InterpGuid,typeof(System.Guid));
			dtTest.Columns.Add(ARTIFICIAL.Interp, typeof(string));

			DataRow drData = dtTest.NewRow();
			drData[TABLES.BloodUnitTest.BloodUnitGuid] = bloodUnitGuid;
			drData[TABLES.BloodTestType.OrderableTestId] = Convert.ToInt32(Common.OrderableTest.AGE);
			drData[TABLES.BloodUnitTest.DivisionCode] = this.TestDivisionCode;
			drData[ARTIFICIAL.InterpGuid] = Guid.NewGuid();
			drData[ARTIFICIAL.Interp] = "P";

			bool retValue = BOL.AntigenTyping.AntigenTypingMismatch(drData,drData[ARTIFICIAL.Interp].ToString());

			Assert.AreEqual(true, retValue);
		}

		[Test]
		public void AntigenTypingMismatch_BloodUnitAntigens_Fail()
		{
			System.Guid bloodUnitGuid = Guid.NewGuid();
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), this._patientGuid, this._bpGuid, true);
			//
			string insertBloodUnitAntigen = "INSERT INTO BloodUnitAntigen (";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.BloodUnitGuid + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.AntigenTypeId + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.Positive + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.DivisionCode + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.RecordStatusCode + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateUser + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateDate + ",";
			insertBloodUnitAntigen += TABLES.BloodUnitAntigen.LastUpdateFunctionId + ")";
			insertBloodUnitAntigen += " VALUES (";
			insertBloodUnitAntigen += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitAntigen += "'20',"; //E antigen
			insertBloodUnitAntigen += "0,"; //Negative
			insertBloodUnitAntigen += "'" + this.TestDivisionCode + "',";
			insertBloodUnitAntigen += "'" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString() + "',";
			insertBloodUnitAntigen += "'" + Common.LogonUser.LogonUserName + "',";
			insertBloodUnitAntigen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitAntigen += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			UnitTestUtil.RunSQL(insertBloodUnitAntigen);
			//
			DataTable dtTest = new DataTable();
			dtTest.Columns.Add(TABLES.BloodTestType.OrderableTestId,typeof(int));
			dtTest.Columns.Add(TABLES.BloodUnitTest.BloodUnitGuid,typeof(System.Guid));
			dtTest.Columns.Add(TABLES.BloodUnitTest.DivisionCode, typeof(string));
			dtTest.Columns.Add(ARTIFICIAL.InterpGuid,typeof(System.Guid));
			dtTest.Columns.Add(ARTIFICIAL.Interp, typeof(string));

			DataRow drData = dtTest.NewRow();
			drData[TABLES.BloodUnitTest.BloodUnitGuid] = bloodUnitGuid;
			drData[TABLES.BloodTestType.OrderableTestId] = Convert.ToInt32(Common.OrderableTest.AGE);
			drData[TABLES.BloodUnitTest.DivisionCode] = this.TestDivisionCode;
			drData[ARTIFICIAL.InterpGuid] = Guid.NewGuid();
			drData[ARTIFICIAL.Interp] = "N";

			bool retValue = BOL.AntigenTyping.AntigenTypingMismatch(drData,drData[ARTIFICIAL.Interp].ToString());

			Assert.AreEqual(false, retValue);
		}

		[Test]
		public void PatientAntigenTypingMismatch_AntibodyIdentified_Pass()
		{
			Guid bloodUnitGuid = Guid.Empty;

			DataTable dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + this.TestDivisionCode + "' ORDER BY LastUpdateDate DESC");
			if (dtBloodUnit.Rows.Count == 0)
			{
				dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + this.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);
			}
			bloodUnitGuid = (Guid)dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

			Guid patientTransfusionRequirementGuid = Guid.NewGuid();
			string sSQLInsertPTR = "INSERT INTO PatientTransfusionRequirement (";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntigenTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			sSQLInsertPTR += "VALUES(";
			sSQLInsertPTR += "'" + patientTransfusionRequirementGuid.ToString() + "',";
			sSQLInsertPTR += "'" + this._patientGuid.ToString() + "',";
			sSQLInsertPTR += "'Anti-C',";
			sSQLInsertPTR += "'" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "',";
			sSQLInsertPTR += "'5',";
			sSQLInsertPTR += "'6',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString() + "',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + this._patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedComponentGuid = Guid.NewGuid();
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
			insertOrderedComponent += "'"+ DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedUnitGuid = Guid.NewGuid();
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid bloodUnitTestGuid1 = Guid.NewGuid();

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			Guid issuedUnitGuid = Guid.NewGuid();
			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(sSQLInsertPTR);
			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			//
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGC), "P");
			Assert.AreEqual(true, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_NegativeResult_Fail()
		{
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(Guid.NewGuid(), Convert.ToInt32(Common.OrderableTest.AGE), "N");
			Assert.AreEqual(false, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_NonIssuedUnit_Fail()
		{
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(Guid.NewGuid(), Convert.ToInt32(Common.OrderableTest.AGc), "P");
			Assert.AreEqual(false, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_AntigenNegativeRequirement_Pass()
		{
			DataTable dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + this.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);
			System.Guid bloodUnitGuid = (Guid)dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

			Guid patientTransfusionRequirementGuid = Guid.NewGuid();
			string sSQLInsertPTR = "INSERT INTO PatientTransfusionRequirement (";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntigenTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			sSQLInsertPTR += "VALUES(";
			sSQLInsertPTR += "'" + patientTransfusionRequirementGuid.ToString() + "',";
			sSQLInsertPTR += "'" + this._patientGuid.ToString() + "',";
			sSQLInsertPTR += "'K',";
			sSQLInsertPTR += "'" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative) + "',";
			sSQLInsertPTR += "'29',";
			sSQLInsertPTR += "'37',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString() + "',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + this._patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedComponentGuid = Guid.NewGuid();
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
			insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedUnitGuid = Guid.NewGuid();
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			Guid issuedUnitGuid = Guid.NewGuid();
			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(sSQLInsertPTR);
			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGK), "P");
			Assert.AreEqual(true, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_WeakD_RhNegative_Pass()
		{
			Guid patientGuid = Guid.NewGuid();
			Guid bloodUnitGuid = Guid.NewGuid();
			Guid patientTreatmentGuid1 = Guid.NewGuid();
			Guid patientTreatmentGuid2 = Guid.NewGuid();
			Guid patientSpecimenGuid = Guid.NewGuid();
			Guid patientOrderGuid1 = Guid.NewGuid();
			Guid patientOrderGuid2 = Guid.NewGuid();
			Guid orderedComponentGuid = Guid.NewGuid();
			Guid orderedTestGuid = Guid.NewGuid();
			Guid orderedUnitGuid = Guid.NewGuid();

			Guid specimenTestGuid1 = Guid.NewGuid();
			Guid specimenTestGuid2 = Guid.NewGuid();
			Guid issuedUnitGuid = Guid.NewGuid();

			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "PatientInsert");

			//This should ensure that the patient is A Neg
			this.BuildABORhFromSpecimenTests(patientGuid, patientTreatmentGuid2, patientSpecimenGuid, patientOrderGuid2, orderedTestGuid, specimenTestGuid1, specimenTestGuid2, "A  ", "N  ");
			//
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), patientGuid, this._bpGuid, true);

			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid1.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid1.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid1.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid1.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			//This is the actual test
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGwD), "P");
			//
			//
			//Now delete all the crap that was just inserted...
			string deleteIssuedUnit = "DELETE FROM IssuedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteIssuedUnit);
			//
			string deleteBloodUnitTests = "DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitTests);
			//
			string deleteSpecimenTests = "DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + patientSpecimenGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteSpecimenTests);
			//
			string deleteOrderedUnit = "DELETE FROM OrderedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedUnit);
			//
			string deleteOrderedComponent = "DELETE FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedComponent);
			//
			string deleteOrderedTest = "DELETE FROM OrderedTest WHERE OrderedTestGuid = '" + orderedTestGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedTest);
			//
			string deletePatientSpecimen = "DELETE FROM PatientSpecimen WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientSpecimen);
			//
			string deletePatientOrder = "DELETE FROM PatientOrder WHERE PatientOrderGuid = '" + patientOrderGuid1.ToString() + "' OR PatientOrderGuid = '" + patientOrderGuid2.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientOrder);
			//
			string deletePatientTreatment = "DELETE FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientTreatment);
			//
			string deleteBloodUnitStatusCodeCurrent = "DELETE FROM BloodUnitStatusCodeCurrent WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatusCodeCurrent);
			//
			string deleteBloodUnitStatus = "DELETE FROM BloodUnitStatus WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatus);
			//
			string deleteBloodUnitFinance = "DELETE FROM BloodUnitFinance WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitFinance);
			//
			string deleteBloodUnitMedia = "DELETE FROM BloodUnitMedia WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitMedia);
			//
			string deleteShipmentCurrent = "DELETE FROM BloodUnitCurrentShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteShipmentCurrent);
			//
			string deleteBloodUnitShipment = "DELETE FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitShipment);
			//
			string deleteBloodUnit = "DELETE FROM BloodUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnit);
			//
			string deletePatient = "DELETE FROM Patient WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatient);
			//
			Assert.AreEqual(true, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_NoTRs_Fail()
		{
			Guid patientGuid = Guid.NewGuid();
			Guid bloodUnitGuid = Guid.NewGuid();
			Guid patientTreatmentGuid = Guid.NewGuid();
			Guid patientOrderGuid = Guid.NewGuid();
			Guid orderedComponentGuid = Guid.NewGuid();
			Guid orderedUnitGuid = Guid.NewGuid();
			Guid issuedUnitGuid = Guid.NewGuid();
			//
			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "Patient Insert");

			//Now issue a unit to them
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), patientGuid, this._bpGuid, true);

			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";


			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGN), "P");
			Assert.AreEqual(false, posResult);
			//
			string deleteIssuedUnit = "DELETE FROM IssuedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteIssuedUnit);
			//
			string deleteBloodUnitTests = "DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitTests);
			//
			string deleteOrderedUnit = "DELETE FROM OrderedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedUnit);
			//
			string deleteOrderedComponent = "DELETE FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedComponent);
			//
			string deletePatientOrder = "DELETE FROM PatientOrder WHERE PatientOrderGuid = '" + patientOrderGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientOrder);
			//
			string deletePatientTreatment = "DELETE FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientTreatment);
			//
			string deleteBloodUnitStatusCodeCurrent = "DELETE FROM BloodUnitStatusCodeCurrent WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatusCodeCurrent);
			//
			string deleteBloodUnitStatus = "DELETE FROM BloodUnitStatus WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatus);
			//
			string deleteBloodUnitFinance = "DELETE FROM BloodUnitFinance WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitFinance);
			//
			string deleteBloodUnitMedia = "DELETE FROM BloodUnitMedia WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitMedia);
			//
			string deleteShipmentCurrent = "DELETE FROM BloodUnitCurrentShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteShipmentCurrent);
			//
			string deleteBloodUnitShipment = "DELETE FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitShipment);
			//
			string deleteBloodUnit = "DELETE FROM BloodUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnit);
			//
			string deletePatient = "DELETE FROM Patient WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatient);
		}

		[Test]
		public void PatientAntigenTypingMismatch_WeakD_RhNegative_Fail()
		{	
			Guid patientGuid = Guid.NewGuid();
			Guid bloodUnitGuid = Guid.NewGuid();
			Guid patientTreatmentGuid1 = Guid.NewGuid();
			Guid patientTreatmentGuid2 = Guid.NewGuid();
			Guid patientSpecimenGuid = Guid.NewGuid();
			Guid patientOrderGuid1 = Guid.NewGuid();
			Guid patientOrderGuid2 = Guid.NewGuid();
			Guid orderedComponentGuid = Guid.NewGuid();
			Guid orderedTestGuid = Guid.NewGuid();
			Guid orderedUnitGuid = Guid.NewGuid();
			Guid specimenTestGuid1 = Guid.NewGuid();
			Guid specimenTestGuid2 = Guid.NewGuid();
			Guid issuedUnitGuid = Guid.NewGuid();

			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "Patient Insert");
			//
			//This should ensure that the patient is A Pos
			this.BuildABORhFromSpecimenTests(patientGuid, patientTreatmentGuid2, patientSpecimenGuid, patientOrderGuid2, orderedTestGuid, specimenTestGuid1, specimenTestGuid2, "A  ", "P  ");
			//
			DataCreator.CreateBloodUnit(bloodUnitGuid, Guid.NewGuid(), patientGuid, this._bpGuid, true);

			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid1.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid1.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid1.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid1.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			//This is the actual test
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGwD), "P");
			//
			//
			//Now delete all the crap that was just inserted...
			string deleteIssuedUnit = "DELETE FROM IssuedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteIssuedUnit);
			//
			string deleteBloodUnitTests = "DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitTests);
			//
			string deleteSpecimenTests = "DELETE FROM SpecimenTest WHERE PatientSpecimenGuid = '" + patientSpecimenGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteSpecimenTests);
			//
			string deleteOrderedUnit = "DELETE FROM OrderedUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedUnit);
			//
			string deleteOrderedComponent = "DELETE FROM OrderedComponent WHERE OrderedComponentGuid = '" + orderedComponentGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedComponent);
			//
			string deleteOrderedTest = "DELETE FROM OrderedTest WHERE OrderedTestGuid = '" + orderedTestGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteOrderedTest);
			//
			string deletePatientSpecimen = "DELETE FROM PatientSpecimen WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientSpecimen);
			//
			string deletePatientOrder = "DELETE FROM PatientOrder WHERE PatientOrderGuid = '" + patientOrderGuid1.ToString() + "' OR PatientOrderGuid = '" + patientOrderGuid2.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientOrder);
			//
			string deletePatientTreatment = "DELETE FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatientTreatment);
			//
			string deleteBloodUnitStatusCodeCurrent = "DELETE FROM BloodUnitStatusCodeCurrent WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatusCodeCurrent);
			//
			string deleteBloodUnitStatus = "DELETE FROM BloodUnitStatus WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitStatus);
			//
			string deleteBloodUnitFinance = "DELETE FROM BloodUnitFinance WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitFinance);
			//
			string deleteBloodUnitMedia = "DELETE FROM BloodUnitMedia WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitMedia);
			//
			string deleteShipmentCurrent = "DELETE FROM BloodUnitCurrentShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteShipmentCurrent);
			//
			string deleteBloodUnitShipment = "DELETE FROM BloodUnitShipment WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnitShipment);
			//
			string deleteBloodUnit = "DELETE FROM BloodUnit WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deleteBloodUnit);
			//
			string deletePatient = "DELETE FROM Patient WHERE PatientGuid = '" + patientGuid.ToString() + "'";
			UnitTestUtil.RunSQL(deletePatient);
			//
			Assert.AreEqual(false, posResult);
		}

		[Test]
		public void PatientAntigenTypingMismatch_AntibodyIdentifiedFromConversion_Pass()
		{
			Guid bloodUnitGuid = Guid.Empty;

			DataTable dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + this.TestDivisionCode + "' ORDER BY LastUpdateDate DESC");
			if (dtBloodUnit.Rows.Count == 0)
			{
				dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM vw_BloodShipAllCurrent WHERE RecordStatusCode = 'A' AND UnitStatusCode IN ('L','A') AND DivisionCode = '" + this.TestDivisionCode + "' ORDER BY LastUpdateDate DESC", true);
			}
			bloodUnitGuid = (Guid)dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

			Guid patientTransfusionRequirementGuid = Guid.NewGuid();
			string sSQLInsertPTR = "INSERT INTO PatientTransfusionRequirement (";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.AntigenTypeId + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			sSQLInsertPTR += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			sSQLInsertPTR += "VALUES(";
			sSQLInsertPTR += "'" + patientTransfusionRequirementGuid.ToString() + "',";
			sSQLInsertPTR += "'" + this._patientGuid.ToString() + "',";
			sSQLInsertPTR += "'Anti-C',";
			sSQLInsertPTR += "'" + Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified) + "',";
			sSQLInsertPTR += "'5',";
			sSQLInsertPTR += "'6',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Converted).ToString() + "',";
			sSQLInsertPTR += "'1',";
			sSQLInsertPTR += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQLInsertPTR += "'" + Common.LogonUser.LogonUserName + "',";
			sSQLInsertPTR += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientTreatmentGuid = Guid.NewGuid();
			string insertPatientTreatment = "INSERT INTO PatientTreatment (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + this._patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid patientOrderGuid = Guid.NewGuid();
			string insertPatientOrder = "INSERT INTO PatientOrder (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + this.TestDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedComponentGuid = Guid.NewGuid();
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'1',";
            insertOrderedComponent += "'" + DataCreator.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += "'2',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + this.TestDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid orderedUnitGuid = Guid.NewGuid();
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + this.TestDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			Guid bloodUnitTestGuid1 = Guid.NewGuid();

			string sSQLInsertBUT1 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 100, "0", orderedComponentGuid, orderedUnitGuid);
			string sSQLInsertBUT2 = this.GetInsertBloodUnitTestSQL(bloodUnitGuid, 102, "C", orderedComponentGuid, orderedUnitGuid);

			Guid issuedUnitGuid = Guid.NewGuid();
			string insertIssuedUnit = "INSERT INTO IssuedUnit (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Common.LogonUser.LogonUserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(sSQLInsertPTR);
			UnitTestUtil.RunSQL(insertPatientTreatment);
			UnitTestUtil.RunSQL(insertPatientOrder);
			UnitTestUtil.RunSQL(insertOrderedComponent);
			UnitTestUtil.RunSQL(insertOrderedUnit);
			UnitTestUtil.RunSQL(sSQLInsertBUT1);
			UnitTestUtil.RunSQL(sSQLInsertBUT2);
			UnitTestUtil.RunSQL(insertIssuedUnit);

			//
			bool posResult = BOL.AntigenTyping.PatientAntigenTypingMismatch(bloodUnitGuid, Convert.ToInt32(Common.OrderableTest.AGC), "P");
			Assert.AreEqual(true, posResult);
		}

		#region Helper Methods

		private void BuildABORhFromSpecimenTests(Guid patientGuid, Guid patientTreatmentGuid, Guid patientSpecimenGuid, Guid patientOrderGuid, Guid orderedTestGuid, Guid specimenTestAboGuid, Guid specimenTestRhGuid, string aboInterp, string rhInterp)
		{
			string [] queries = new string [6];
			StringBuilder query = new StringBuilder();
			//
			query.Append("INSERT INTO PatientTreatment (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[0] = query.ToString();
			query.Remove(0, query.Length);
			//
			query.Append("INSERT INTO PatientOrder (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentGuid.ToString()).Append("',");
			query.Append("'").Append(this.TestDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[1] = query.ToString();
			query.Remove(0, query.Length);
			//
			query.Append("INSERT INTO PatientSpecimen (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenReceivedDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenExpirationDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddDays(3))).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(DateTime.Now.ToString("yyyyMMddHHmmss")).Append("',");
			query.Append("'").Append(this.TestDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[2] = query.ToString();
			query.Remove(0, query.Length);
			//
			query.Append("INSERT INTO OrderedTest (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenGuid.ToString()).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.OrderableTest.ABORh)).Append("',");
			query.Append("'C',");
			query.Append("'C',");
			query.Append("'").Append(this.TestDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[3] = query.ToString();
			query.Remove(0, query.Length);
			//
			query.Append("INSERT INTO SpecimenTest (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(aboInterp).Append("',");
			query.Append("'A',");
			query.Append("'").Append(this.TestDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[4] = query.ToString();
			query.Remove(0, query.Length);
			//
			query.Append("INSERT INTO SpecimenTest (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(rhInterp).Append("',");
			query.Append("'A',");
			query.Append("'").Append(this.TestDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			queries[5] = query.ToString();
			query.Remove(0, query.Length);
			//
			for (int idx = 0; idx < queries.Length; idx++)
			{
				UnitTestUtil.RunSQL(queries[idx]);
			}
		}


		/// <summary>
		/// Code reuse -- Creates the SQL for a BloodUnitTest record insert
		/// </summary>
		/// <param name="buGuid"></param>
		/// <param name="testTypeId"></param>
		/// <param name="testResultId"></param>
		/// <param name="ocGuid"></param>
		/// <param name="ouGuid"></param>
		/// <returns></returns>
		private string GetInsertBloodUnitTestSQL(System.Guid buGuid, int testTypeId, string testResultId, System.Guid ocGuid, System.Guid ouGuid)
		{

			string sSQL = "INSERT INTO BloodUnitTest (";
			sSQL += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			sSQL += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			sSQL += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			sSQL += TABLES.BloodUnitTest.TestDate + ",";
			sSQL += TABLES.BloodUnitTest.TestTechId + ",";
			sSQL += TABLES.BloodUnitTest.TestResultId + ",";
			sSQL += TABLES.BloodUnitTest.EntryTechId + ",";
			sSQL += TABLES.BloodUnitTest.EntryMethodCode + ",";
			sSQL += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			sSQL += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			sSQL += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			sSQL += TABLES.BloodUnitTest.RecordStatusCode + ",";
			sSQL += TABLES.BloodUnitTest.DivisionCode + ",";
			sSQL += TABLES.BloodUnitTest.LastUpdateDate + ",";
			sSQL += TABLES.BloodUnitTest.LastUpdateUser + ",";
			sSQL += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			sSQL += "VALUES (";
			sSQL += "'" + Guid.NewGuid().ToString() + "',";
			sSQL += "'" + buGuid.ToString() + "',";
			sSQL += "'" + testTypeId.ToString() + "',";
			sSQL += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQL += "'" + Common.LogonUser.LogonUserName + "',";
			sSQL += "'" + testResultId + "',";
			sSQL += "'" + Common.LogonUser.LogonUserName + "',";
			sSQL += "'M',";
			sSQL += "'" + ocGuid.ToString() + "',";
			sSQL += "'" + ouGuid.ToString() + "',";
			sSQL += "'2',";
			sSQL += "'A',";
			sSQL += "'" + this.TestDivisionCode + "',";
			sSQL += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			sSQL += "'" + Common.LogonUser.LogonUserName + "',";
			sSQL += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";


			return(sSQL);

		}

		#endregion

	}
}
#endif
